﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.UI;
using TMPro;
using UnityEngine.EventSystems;
using System;

[System.Serializable]
public class IncrementerBox : Selectable, IPointerEnterHandler, IPointerExitHandler
{
    //Event in editor to allow code to run when selected
    [SerializeField]
    private Button.ButtonClickedEvent _OnClick = new Button.ButtonClickedEvent();

    public Button.ButtonClickedEvent onClick
    {
        get
        {
            return _OnClick;
        }

        set
        {
            _OnClick = value;
        }
    }

    #region EventArgs Classes
    public class CurrentValueChangedEventArgs : EventArgs
    {
        public int OldValue { get; private set; }
        public int NewValue { get; private set; }

        public CurrentValueChangedEventArgs(int oldValue, int newValue)
        {
            OldValue = oldValue;
            NewValue = newValue;
        }
    }

    public event EventHandler<CurrentValueChangedEventArgs> CurrentValueChanged;    //Called when the current value changes, sends custom event args
    #endregion

    //Public Properties
    [SerializeField]
    public int IncrementAmount = 1; //How much to increment buy
    public int MinimumAmount = 0;
    public int MaximumAmount = 10;
    public int HeldDelay = 1000;    //Delay between holding the button causing acceleration to begin
    public int HeldAccelerationDelta = 150; //How much time between changing values during acceleration

    //UI elements
    public CustomButton MinusButton;
    public TMP_Text CurrentValueText;
    public CustomButton PlusButton;

    private float _TimeButtonHeld;
    private float _TimeSinceLastHeldPress;
    private Animator _Animator;

    public int CurrentValue { get; private set; } = 0;

    protected override void Start()
    {
        base.Start();
        CurrentValueText.text = CurrentValue.ToString();

        //Get components, subscribe to events
        MinusButton.Button.onClick.AddListener(OnMinusButtonClicked);
        PlusButton.Button.onClick.AddListener(OnPlusButtonClicked);
        _Animator = GetComponent<Animator>();
    }

    private void Update()
    {
        if(MinusButton.IsPressed)
        {
            _TimeButtonHeld += (Time.unscaledDeltaTime * 1000.0f);
            if (_TimeButtonHeld >= HeldDelay)
            {
                //Time to accelerate
                _TimeSinceLastHeldPress += (Time.unscaledDeltaTime * 1000.0f);

                if(_TimeSinceLastHeldPress >= HeldAccelerationDelta)
                {
                    //Time to update to the new value
                    _TimeSinceLastHeldPress = 0.0f;
                    MinusButton.OnClick();
                    MinusButton.Button.onClick.Invoke();
                }
            }
        }

        else if(PlusButton.IsPressed)
        {
            _TimeButtonHeld += (Time.unscaledDeltaTime * 1000.0f);
            if (_TimeButtonHeld >= HeldDelay)
            {
                //Time to accelerate
                _TimeSinceLastHeldPress += (Time.unscaledDeltaTime * 1000.0f);

                if (_TimeSinceLastHeldPress >= HeldAccelerationDelta)
                {
                    //Time to update to the new value
                    _TimeSinceLastHeldPress = 0.0f;
                    PlusButton.OnClick();
                    PlusButton.Button.onClick.Invoke();
                }
            }
        }

        else
        {
            //Nothing's held, reset all acceleration values
            _TimeButtonHeld = 0.0f;
            _TimeSinceLastHeldPress = 0.0f;
        }
    }

    /// <summary>
    /// Sets the current value if within range and raises the event
    /// </summary>
    /// <param name="val">The value to become current</param>
    public void SetCurrentValue(int val)
    {
        if (val >= MinimumAmount && val <= MaximumAmount)
        {
            //In range, raise the event and set the value
            OnCurrentValueChanged(new CurrentValueChangedEventArgs(CurrentValue, val));
            CurrentValue = val;
            CurrentValueText.text = CurrentValue.ToString();
        }
    }

    /// <summary>
    /// Handler for the minus button being clicked, tries to decrement the current value if within range and raises the event
    /// </summary>
    private void OnMinusButtonClicked()
    {
        if(CurrentValue - IncrementAmount >= MinimumAmount)
        {
            //In range, raise the event and set the value
            OnCurrentValueChanged(new CurrentValueChangedEventArgs(CurrentValue, CurrentValue - IncrementAmount));
            CurrentValue -= IncrementAmount;
            CurrentValueText.text = CurrentValue.ToString();
        }
    }

    /// <summary>
    /// Handler for the plus button being clicked, tries to increment the current value if within range and raises the event
    /// </summary>
    private void OnPlusButtonClicked()
    {
        if (CurrentValue + IncrementAmount <= MaximumAmount)
        {
            //In range, raise the event and set the value
            OnCurrentValueChanged(new CurrentValueChangedEventArgs(CurrentValue, CurrentValue + IncrementAmount));
            CurrentValue += IncrementAmount;
            CurrentValueText.text = CurrentValue.ToString();
        }
    }

    /// <summary>
    /// Caller for raising the event
    /// </summary>
    /// <param name="e">EventArgs to be sent with the event</param>
    private void OnCurrentValueChanged(CurrentValueChangedEventArgs e)
    {
        //Called when the selected item changes and sends an event notification to all subscribers
        EventHandler<CurrentValueChangedEventArgs> handler = CurrentValueChanged;
        if (handler != null)
        {
            handler(this, e);
        }
    }
}
